unit Rectangle;
{*******************************************************************************
  ShapesDemo
  Written by David Clegg, davidclegg@optusnet.com.au.

  This unit contains the Rectangle class used to render a rectangle onto a GDI+
  drawing surface.
*******************************************************************************}

interface

uses
  Shape, System.Drawing, System.Drawing.Drawing2D;

type

	/// <summary>
	/// Class to draw a rectangle.
	/// </summary>
  TRectangle = class(TShape)
  protected
    function GetShape(pStartPoint, pEndPoint: Point): GraphicsPath; override;
  public
    procedure DrawRectangle(pBrush: Brush; pStartPoint, pEndPoint: Point);
    procedure OutlineRectangle(pPen: Pen; pStartPoint, pEndPoint: Point);
  end;

implementation

/// <summary>
/// Alternative method to draw a solid rectangle. Not as good as the
/// inherited Draw method as it doesn't cater for if an endPoint X or Y
/// value is less than its corresponding startPoint value.
/// </summary>
procedure TRectangle.DrawRectangle(pBrush: Brush; pStartPoint, pEndPoint: Point);
begin
  Canvas.FillRectangle(pBrush, DoGetRectangle(pStartPoint, pEndPoint));
end;

/// <summary>
/// Alternative method to draw a rectangle outline. Not as good as the
/// inherited Draw method as it doesn't cater for if an endPoint X or Y
/// value is less than its corresponding startPoint value.
/// </summary>
procedure TRectangle.OutlineRectangle(pPen: Pen; pStartPoint, pEndPoint: Point);
begin
  Canvas.DrawRectangle(pPen, DoGetRectangle(pStartPoint, pEndPoint));
end;

/// <summary>
/// Return a GraphicsPath representing the bounds of the Rectangle based on
/// pStartPoint and pEndPoint.
/// </summary>
function TRectangle.GetShape(pStartPoint, pEndPoint: Point): GraphicsPath;
var
  lTopRight: Point;
  lBottomLeft: Point;
begin
  Result := GraphicsPath.Create;

  lTopRight := Point.Create(pStartPoint.X, pEndPoint.Y);
  lBottomLeft := Point.Create(pEndPoint.X, pStartPoint.Y);

  Result.AddLine(pStartPoint, lTopRight);
  Result.AddLine(lTopRight, pEndPoint);
  Result.AddLine(pEndPoint, lBottomLeft);
  Result.AddLine(lBottomLeft, pStartPoint);
end;

end.
